#include "stdafx.h"
#include <string.h>
#include <stdlib.h>
#include "sqxrunner.h"

/* 
If you have a QM (or QM+) license you will need to uncomment the define below so the application
puts itself in Prediction mode instead of in model mode. 
*/
/* #define SIMCA_QM */

#ifdef SIMCA_QM 
#include "SQMCInterface.h"
#endif

/**
*  Do a prediction
* @param pObj           In: The object representing the project to get the data from.
* @param pOut           In: The stream to print the results to.
* @param pErr           In: The stream to print possible errors to.
* @param nModelNumber   In: The number of the model to do the predictions for.
* @param pObservations  In: The data to predict.
* @param pQualitative   In: The qualitative data to do predictions on.
                            If the model doesn't have any qualitative variables, 
                            set this parameter to NULL.
* @param bIsCompressed  In: Set to 1 if the project contains compressed data and
                            the compression is taken care of outside SIMCA-QM.
*/
static int SQXRunner_DoPredictions(SQXRunner* pObj,
                                   FILE* pOut, 
                                   FILE* pErr,
                                   int nModelNumber,
                                   SQP_ObservationRawData *pObservations, 
                                   SQP_QualitativeRawData *pQualitative, 
                                   int bIsCompressed);

/**
*  Get the result after a prediction has been done.
* @param pObj           In: The object representing the project to get the data from.
* @param pOut           In: The stream to print the results to.
* @param pErr           In: The stream to print possible errors to.
* @param pHandle        In: The prediction handle to get the predicted data from.
* @param nModelNumber   In: The number of the model to get the predicted data from.
*/
static void SQXRunner_GetResults(SQXRunner* pObj,
                                 FILE* pOut, 
                                 FILE* pErr,
                                 SQX_PredictionHandle pHandle, 
                                 int nModelNumber); 

/**
*  Get data from the model.
* @param pObj           In: The object representing the project to get the data from.
* @param pOut           In: The stream to print the results to.
* @param pErr           In: The stream to print possible errors to.
* @param nModelNumber   In: The number of the model to get the data from.
*/
static int SQXRunner_GetModelParameters(SQXRunner* pObj,
                                        FILE* pOut,
                                        FILE* pErr,
                                        int nModelNumber);

/**
 *	Print the matrix tab separated to a file.
 * @param pMatrix In: The matrix to be printed.
 * @param pOut    In: The stream to print the matrix to.
 * @param pErr    In: The stream to print possible errors to.
 */
static void SQXRunner_PrintSQX_FloatMatrix(SQX_FloatMatrix *pMatrix,
                                         FILE* pOut,
                                         FILE* pErr);

int SQXRunner_Init(SQXRunner* pObj, char* szUSPName, char* szPredictionFileName)
{
#ifdef _WIN32
   pObj->mszUSPName = strdup(szUSPName);
#else
   pObj->mszUSPName = (char*)strdup(szUSPName);
#endif
   if (szPredictionFileName != NULL)
     pObj->mInputFile = fopen(szPredictionFileName, "r");
   else
      pObj->mInputFile = NULL;

   pObj->szFileBuffer = NULL;
   pObj->iFileNumCols = 0;
   pObj->iFileNumRows = 0;

   return -1;
}

void SQXRunner_Destroy(SQXRunner* pObj)
{
   free(pObj->mszUSPName);
   if (pObj->szFileBuffer)
   {
      SQX_ClearStringVector(&pObj->mstrVariableNames);
      free(pObj->szFileBuffer);
   }
   if (pObj->mInputFile)
      fclose(pObj->mInputFile);
}

void SQXRunner_Run(SQXRunner* pObj, FILE* pOut, FILE* pErr) 
{
   int     iNumVar;                 /* Number of variables */
   int     iNumLags;                /* Number of lagged x variables */
   int     iNumQual;                /* Number of qualitative variables */
   int  	  iSettings;               /* Number of settings for a qualitative variable */
   int     iModelNumber;            /* Model number */
   int     iNumLagParents;          /* Number of lag parents */
   int     iNumQualLagParents;      /* Number of qualitative lag parents */
   int     iNumModels = 0;          /* Number of models in the project */
   int     iLags;                   /* Will contain the size of piLagSteps */
   int     iLagStep;                /* How many steps a variable is to be lagged */
   int     iIsFitted;               /* Variable to inform if the model is fitted or not.*/

   int      iModelIndex;            /* Model index */
   int      nIx;                    /* Index used in loops. */
   char*    szErrString;            /* String that will contain possible error. */
   int      iNumQualNames;
   int      iNumObs = 10;           /* Number of observations to predict.*/
   int      iObs;                   /* Used as index. */
 
   float   fRawData;                /* Counter for the data that will be send to do predictions on. */
   float   fLagData = 0;            /* Counter for the lagged data that will be send to do predictions on. */
   char    *szString = NULL;        /* Will contain char ptr returned from calls to S-QP */
   char    *szModelName;            /* Will contain a model name */
   char    *pszMotherLag;           /* Will contain a name for a lagged mother variable. */
   char    *szProjectName = NULL;   /* Will contain the project name */
   char    *pszQualVarName = NULL;  /* Will contain the name of a qualitative variable */
   const char* szConstString;  
   int     iCompressed = 0;         /* Flag that indicates that we never send compressed
                                       data to SIMCA-QM. If compression needs to be done we
                                       let SIMCA-QM handle it. */
   int iRecover=0;                  /* 1 if we can recover the usp from temporary directory.*/
   char szTempName[1024];           /* Buffer for the temporary directory. */
#ifdef SIMCA_QM 
   SQM_DllMode eMode;
#endif

   SQX_StringVector  oXVarName;              /* Names of the x variables in a model */
   SQX_StringVector  oQualNames;             /* Names of the qualitative variables in a model */
   SQX_StringVector	oNamesSettings;         /* Holds the settings for a given qualitative variable. */
   SQX_FloatMatrix*  pRawData 	= NULL;     /* The raw data that will be used to do prediction on. */
   SQX_FloatMatrix*  pLagData 	= NULL;     /* The lagged data that will be used to do prediction on. */
   SQX_StringMatrix* pQualData   	= NULL;  /* The qualitative data that will be used to do prediction on. */
   SQX_StringMatrix* pQualLagData	= NULL;  /* The lagged qualitative data that will be used to do prediction on. */
   SQX_IntVector     oLagSteps;              /* vector that will contain how many steps to lag. */
   SQX_FloatMatrix   oCompData;              /* matrix that will contain the compressed data */
   /* A structure that will be send to Predict() containing the data 
   that will be used for prediction. */
   SQP_ObservationRawData* pObservationRawData = NULL; 
   /* A structure that will be send to Predict() containing the qualitative data 
   that will be used for prediction. */
   SQP_QualitativeRawData* pQualitativeRawData = NULL;

   /* Read variable names from prediction file if any */
   if (pObj->mInputFile)
   {
      if (!UmFileReader(pObj, pErr))
      {
         fprintf(pErr, "Failed to read the prediction file.\n");
         return;
      }
      iNumObs = pObj->iFileNumRows;
   }
   /* This is the file we want to use in SIMCA-QM */
   fprintf(pOut, "Path to usp file:        %s\n",pObj->mszUSPName);

   /* Get temporary directory. */
   tmpnam(szTempName);
   if (szTempName!=NULL)
   {
      /* Get directory. */
      int iLast=(int)strlen(szTempName);
      while(iLast>0)
      {
#ifdef _WIN32
         if(szTempName[iLast-1] == '\\')
#else
         if(szTempName[iLast-1] == '/')
#endif
         {/* Found a separator. */
            szTempName[iLast]='\0';
            break;
         }
         iLast--;
      }
   }

   if (!SQX_AddProject(pObj->mszUSPName, 1, NULL /* Not a password protected project*/, &pObj->mszProjID))
   {
      szErrString = "Could not open project. AddProject failed.";
      goto error_exit;
   }

#ifdef SIMCA_QM 
   /* For SIMCA-QM and SIMCA-QM+ there exist two modes, Modeling and Prediction mode
      when a project is added the default mode is modeling, but to make a prediction
      the mode must be switched to Prediction. */
   if (!SQM_GetMode(pObj->mszProjID, &eMode))
   {
      szErrString = "Could not switch to prediction mode";
      goto error_exit;
   }
   if (eMode == SQM_Modeling)
   {
      /* Change mode to prediction before requesting predictions. */
      if (!SQM_ChangeMode(&pObj->mszProjID, SQM_Prediction))
      {
         szErrString = "Could not switch to prediction mode";
         goto error_exit;
      }
   }
#endif

   /*
      Get the project name
   */
   if (!SQX_GetProjectName(pObj->mszProjID, &szProjectName))
   {
      szErrString = "GetProjectName failed.";
      goto error_exit;
   }
   /* pOut is a temp file where we store the results from our predictions */
   fprintf(pOut, "Project name:            %s\n",szProjectName);

   /*
      Get number of fitted models
   */
   if (!SQX_GetNumberOfModels(pObj->mszProjID, &iNumModels))
   {
      szErrString = "GetNumberOfModels failed.";
      goto error_exit;
   }
   fprintf(pOut, "Number of models: %d\n\n",iNumModels);

   /*
      Get info for each one of the fitted models.
   */
   for(iModelIndex = 1; iModelIndex <= iNumModels; ++iModelIndex)
   {
      SQX_StringVector oCompVarName; 

      /*
         Get the model number connected with this index.
      */
      if (!SQX_GetModelNumber(pObj->mszProjID, iModelIndex, &iModelNumber))
      {
         szErrString = "GetModelNumber failed.";
         goto error_exit;
      }

      if (!SQX_IsModelFitted(pObj->mszProjID, iModelNumber, &iIsFitted))
      {
         szErrString = "IsModelFitted failed.";
         goto error_exit;
      }

      if(iIsFitted != 1)
      {
         /*
         The model is not fitted, go to the next model.
         */
         fprintf(pOut, "Model is not fitted.\n");
      }
      else
      {
         /*
         Get the name of the model connected with this number.

         Note: Model number is input NOT model index from now on.
         */
         if (!SQX_GetModelName(pObj->mszProjID, iModelNumber, &szModelName))
         {
            szErrString = "GetModelNumberFromIndex failed.";
            goto error_exit;
         }
         fprintf(pOut, "==================================\n");
         fprintf(pOut, "Model name: %s\n",szModelName);

         /* Get the variable names needed for compression, if there are any. */
         if(!SQP_GetVarNamesForCompress(pObj->mszProjID, &oCompVarName))
         {
            szErrString = "GetVarNamesForCompress failed.";
            goto error_exit;
         }
         else
         {
            int iNumCompVar = SQX_GetNumStringsInVector(&oCompVarName);
            if(iNumCompVar > 0)
            {
               /* Yes, there are "ordinary" variables in this model. */
               SQX_FloatMatrix *pCompRawData = (SQX_FloatMatrix*)malloc(sizeof(SQX_FloatMatrix));
               if (pObj->mInputFile) 
               {
                  /* if we have a datafile to read from */
                  if (!GetQuantitativeData(pObj, &oCompVarName, pErr, *&pCompRawData))
                  {
                     SQX_ClearStringVector(&oCompVarName);
                     free(pCompRawData);
                     pCompRawData = NULL;
                     szErrString = "GetQuantitativeData failed.";
                     goto error_exit;
                  }
               }
               else if (!SQX_InitFloatMatrix(pCompRawData, iNumObs, iNumCompVar))
               {
                  /*
                  For this sample we just set 1, 2, 3,......
                  You should set your own data, read  e.g.  from a
                  file or a database.
                  */
                  fRawData = 0;
                  for(iObs = 0; iObs < iNumObs; ++iObs)
                  {
                     for(nIx = 0; nIx < iNumCompVar; ++nIx)
                     {
                        if (!SQX_SetDataInFloatMatrix(pCompRawData, iObs + 1, nIx + 1, ++fRawData))
                        {
                           szErrString = "SQX_SetDataInFloatMatrix failed.";
                           goto error_exit;
                        }
                     }
                  }
               }
               if(!SQP_Compress(pObj->mszProjID, pCompRawData, &oCompVarName, 1, &oCompData))
               {
                  szErrString = "Compress failed.";
                  goto error_exit;
               }
               else
               {
                  fprintf(pOut, "Compressed matrix:\n");
                  SQXRunner_PrintSQX_FloatMatrix(&oCompData, pOut, pErr);
                  SQX_ClearFloatMatrix(&oCompData);
               }
               SQX_ClearStringVector(&oCompVarName);
               SQX_ClearFloatMatrix(pCompRawData);
               free(pCompRawData);
            }
         }

         /*
         Get the names of the x variables in the model. 
         These are the names of the variables who's data are needed
         as input for a prediction.
         */
         if (!SQP_GetQuantitativeNamesForPredict(pObj->mszProjID, iModelNumber, iCompressed , &oXVarName))
         {                        
            szErrString = "GetQuantitativeNamesForPredict failed.";
            goto error_exit;
         }
         else
         {
            /*
            Set the data to use for the Prediction.
            */
            iNumVar = SQX_GetNumStringsInVector(&oXVarName);
            if(iNumVar > 0)
            {
               /* Yes, there are "ordinary" variables in this model. */
               pRawData = (SQX_FloatMatrix*)malloc(sizeof(SQX_FloatMatrix));
               if (pObj->mInputFile)
               {  
                  /* If we have a datafile to read from */
                  if (!GetQuantitativeData(pObj, &oXVarName, pErr, *&pRawData))
                  {
                     SQX_ClearStringVector(&oXVarName);
                     free(pRawData);
                     pRawData = NULL;
                     szErrString = "GetQuantitativeData failed.";
                     goto error_exit;
                  }
               }
               else if (!SQX_InitFloatMatrix(pRawData, iNumObs, iNumVar))
               {                        
                  szErrString = "SQX_InitFloatMatrix failed.";
                  goto error_exit;
               }
               else
               {
                  {
                     /*
                     For this sample we just set 1, 2, 3,......
                     You should set your own data, read  e.g.  from a
                     file or a database.
                     */
                     fRawData = 0;
                     for(iObs = 0; iObs < iNumObs; ++iObs)
                     {
                        for(nIx = 0; nIx < iNumVar; ++nIx)
                        {
                           if (!SQX_SetDataInFloatMatrix(pRawData, iObs + 1, nIx + 1, ++fRawData))
                           {                        
                              szErrString = "SQX_SetDataInFloatMatrix failed.";
                              goto error_exit;
                           }
                        }
                     }
                  }
               }
            }
         }
         
         /*
         Check if the project contains any lagged data.
         These are the names of the lagged variables who's data are needed
         as input for a prediction.
         */
         if (!pObj->mInputFile)
         {
            if (!SQP_GetNumberOfLagParents(pObj->mszProjID, iModelNumber, 0 /*bQualitative*/, &iNumLagParents))
            {                        
               szErrString = "GetNumberOfLagParents failed.";
               goto error_exit;
            }

            if(iNumLagParents > 0)
            {
               /* Yes, there are lagged variables in this model. */
               int iLagVar;
               pLagData = (SQX_FloatMatrix*)malloc(sizeof(SQX_FloatMatrix));
               /* Check how many lagged variables there are so that we can allocate
               memory for the lagged data. */
               if (!SQP_GetNumberOfLags(pObj->mszProjID, iModelNumber, 0 /*bQualitative*/, &iNumLags))
               {                        
                  szErrString = "GetNumberOfLags failed.";
                  goto error_exit;
               }
               if (!SQX_InitFloatMatrix(pLagData, iNumObs, iNumLags))
               {                        
                  szErrString = "SQX_InitFloatMatrix failed.";
                  goto error_exit;
               }
               else
               {
                  int iLagOffset = 0;
                  for(nIx = 0; nIx < iNumLagParents; ++nIx)
                  {
                     if (!SQP_GetVariableLagInfo(pObj->mszProjID, iModelNumber, nIx + 1, 1 /*nVarID*/, 0 /*bQualitative*/, &pszMotherLag, &oLagSteps))
                     {                        
                        szErrString = "GetVariableLagInfo failed.";
                        goto error_exit;
                     }
                     else
                     {
                        /*
                        Now you have the name of the lagged mother variable in pszMotherLag
                        so that you can get data for that variable from your file/DB.
                        This "mothervariable" has iLags number of lagged variable
                        as a part of the model.*/

                        /*For each lagged variable check how many steps it should
                        be lagged.*/
                        iLags = SQX_GetIntVectorSize(&oLagSteps);
                        for(iObs = 0; iObs < iNumObs; ++iObs)
                        {
                           for(iLagVar=0; iLagVar<iLags; ++iLagVar)
                           {
                              SQX_GetDataFromIntVector(&oLagSteps, iLagVar+1, &iLagStep);
                              /*
                              Now you can get the data that is iLagStep's "old", for the 
                              pszMotherLag variable, from your file/DB.

                              For this sample we just set 1, 2, 3,......
                              You should set your own data, read  e.g.  from a
                              file or a database.
                              */
                              if (!SQX_SetDataInFloatMatrix(pLagData, iObs + 1, iLagOffset + iLagVar + 1, ++fLagData))
                              {                        
                                 szErrString = "SQX_SetDataInFloatMatrix failed.";
                                 goto error_exit;
                              }
                           }
                        }
                        iLagOffset += iLags;
                        SQX_ClearIntVector(&oLagSteps);
                     }
                  }
               }
            }
         }
         /*
         Check if the project contains any qualitative data.
         These are the names of the qualitative variables whos data are 
         needed as input for a prediction.
         */
         if (!SQP_GetQualitativeNamesForPredict(pObj->mszProjID, iModelNumber, &oQualNames))
         {                        
            szErrString = "GetQualitativeNamesForPredict failed.";
            goto error_exit;
         }

         iNumQual = SQX_GetNumStringsInVector(&oQualNames);
         if (iNumQual > 0)
         {
            /* Yes, there are qualitative variables in this model. */
            pQualData = (SQX_StringMatrix*)malloc(sizeof(SQX_StringMatrix));
            if (pObj->mInputFile)
            {
               /* If we have a datafile to read from */
               if (!GetQualitativeData(pObj, &oQualNames, pErr, *&pQualData))
               {
                  SQX_ClearStringVector(&oQualNames);
                  free(pQualData);
                  pQualData = NULL;
                  szErrString = "GetQualitativeData failed.";
                  goto error_exit;
               }
            }
            else if (!SQX_InitStringMatrix(pQualData, iNumObs, iNumQual))
            {                        
               szErrString = "SQX_InitStringMatrix failed.";
               goto error_exit;
            }
            else
            {
               /*
               For this sample we just set a valid value, whichever....
               You should set your own data, read e.g. from a
               file or a database.
               */
               for(nIx = 1; nIx <= iNumQual; nIx++)
               {
                  /* Get the settings for this qualitative variables so that we can set a valid value for this sample. */
                  if (!SQP_GetQualitativeInfo(pObj->mszProjID, iModelNumber, nIx, &pszQualVarName, &iSettings, &oNamesSettings))
                  {                        
                     szErrString = "GetQualitativeInfo failed.";
                     goto error_exit;
                  }

                  if (!SQX_GetStringFromVector(&oNamesSettings, 1, &szConstString))
                  {                        
                     szErrString = "GetStringFromMatrix failed.";
                     goto error_exit;
                  }
                  for(iObs = 0; iObs < iNumObs; ++iObs)
                  {
                     /* Set the value for this qualitative variable. */
                     if (!SQX_SetStringInMatrix(pQualData, iObs + 1, nIx, szConstString))
                     {                        
                        szErrString = "SetStringInMatrix failed.";
                        goto error_exit;
                     }
                  }
                  SQX_ClearStringVector(&oNamesSettings);
               }
            }
         }

         /*
         Check if the project contains any qualitative data that are lagged.
         These are the names of the lagged qualitative variables whos data are 
         needed as input for a prediction.
         */
         if (!pObj->mInputFile)
         {
            if (!SQP_GetNumberOfLagParents(pObj->mszProjID, iModelNumber, 1 /*bQualitative*/, &iNumQualLagParents))
            {                        
               szErrString = "GetNumberOfLagParents failed.";
               goto error_exit;
            }

            if(iNumQualLagParents > 0)
            {
               int iNumQualLags;
               /* Yes, there are lagged variables in this model. */
               pQualLagData = (SQX_StringMatrix*)malloc(sizeof(SQX_StringMatrix));
               /* Check how many lagged variables there are so that we can allocate
               memory for the lagged data. */
               if (!SQP_GetNumberOfLags(pObj->mszProjID, iModelNumber, 1 /*bQualitative*/, &iNumQualLags))
               {                        
                  szErrString = "GetNumberOfLags failed.";
                  goto error_exit;
               }
               if (!SQX_InitStringMatrix(pQualLagData, iNumObs, iNumQualLags))
               {                        
                  szErrString = "SQX_InitStringMatrix failed.";
                  goto error_exit;
               }
               else
               {
                  int iLagOffset = 0;
                  for(nIx = 1; nIx <= iNumQualLagParents; ++nIx)
                  {
                     if (!SQP_GetVariableLagInfo(pObj->mszProjID, iModelNumber, nIx, 1 /*nVarID*/, 1 /*bQualitative*/, &pszMotherLag, &oLagSteps))
                     {                        
                        szErrString = "GetVariableLagInfo failed.";
                        goto error_exit;
                     }
                     else
                     {
                        int iLagVar;
                        /*
                        Now you have the name of the lagged mother variable in pszMotherLag
                        so that you can get data for that variable from your file/DB.
                        This "mothervariable" has iLags number of lagged variable
                        as a part of the model.

                        For each lagged variable check how many steps it should
                        be lagged.
                        */
                        /* Get the settings for this qualitative variables so that we can set a valid value for this sample. */
                        if (!SQP_GetQualitativeInfo(pObj->mszProjID, iModelNumber, nIx, &pszQualVarName, &iSettings, &oNamesSettings))
                        {                        
                           szErrString = "GetQualitativeInfo failed.";
                           goto error_exit;
                        }

                        /* Get the first string in the vector.*/
                        if (!SQX_GetStringFromVector(&oNamesSettings, 1, &szConstString))
                        {                        
                           szErrString = "SQX_GetStringFromVector failed.";
                           goto error_exit;
                        }

                        /* For each lagged variable check how many steps it should
                        be lagged.*/
                        iLags = SQX_GetIntVectorSize(&oLagSteps);

                        for(iLagVar=0; iLagVar<iLags; ++iLagVar)
                        {
                           /*iLagStep = piLagSteps[iLagVar];*/
                           SQX_GetDataFromIntVector(&oLagSteps, iLagVar+1, &iLagStep);

                           for(iObs = 0; iObs < iNumObs; ++iObs)
                           {
                              /*
                              Now you can get the data that is iLagStep's "old",
                              for the qualitative variable pszMotherLag, from your file/DB.

                              For this sample we just set a valid value, whichever...
                              You should set your own data, read e.g. from a
                              file or a database.
                              */

                              if (!SQX_SetStringInMatrix(pQualLagData, iObs + 1, iLagOffset + iLagVar + 1, szConstString))
                              {                        
                                 szErrString = "SetStringInMatrix failed.";
                                 goto error_exit;
                              }
                           }
                        }
                        iLagOffset += iLags;
                        SQX_ClearStringVector(&oNamesSettings);
                        SQX_ClearIntVector(&oLagSteps);
                     }
                  }
               }
            }
         }      
         /*
         Print the names of the variables in the file.
         */
         iNumVar = SQX_GetNumStringsInVector(&oXVarName);
         if (iNumVar > 0)
         {
            int iColIter;
            fprintf(pOut, "X variable names:\n");
            for (iColIter = 0; iColIter < iNumVar; ++iColIter)
            {
               if (!SQX_GetStringFromVector(&oXVarName, iColIter + 1, &szConstString))
               {                        
                  szErrString = "SQX_GetStringFromVector failed.";
                  goto error_exit;
               }

               if (szConstString != NULL)
               {
                  /* Print the variable name. */
                  fprintf(pOut, "%s\t", szConstString);
               }
            }
            fprintf(pOut, "\n");
         }
         SQX_ClearStringVector(&oXVarName);
         iNumQualNames = SQX_GetNumStringsInVector(&oQualNames);
         if (iNumQualNames > 0)
         {
            int iColIter;
            fprintf(pOut, "Qualitative variable names:\n");

            for (iColIter = 0; iColIter < iNumQualNames; ++iColIter)
            {
               if (!SQX_GetStringFromVector(&oQualNames, iColIter + 1, &szConstString))
               {                        
                  szErrString = "SQX_GetStringFromVector failed.";
                  goto error_exit;
               }

               if (szConstString != NULL)
               {
                  /* Print the qualitative variable name. */
                  fprintf(pOut, "%s\t",szConstString);
               }
            }
            fprintf(pOut, "\n");
         }
         SQX_ClearStringVector(&oQualNames);

         /*
         Store the "ordinary" observation data in the
         container that is sent to S-QP
         */
         if ((pRawData != NULL) || (pLagData != NULL))
         {
            pObservationRawData = (SQP_ObservationRawData*)malloc(sizeof(SQP_ObservationRawData));

            pObservationRawData->pRawData  = pRawData;
            pObservationRawData->pVarNames    = NULL;
            pObservationRawData->pLagData  = pLagData;
            pObservationRawData->pLagVarNames = NULL;

         }

         /*
         Store the qualitative observation data in the
         container that are sent to S-QP
         */

         if ((pQualData != NULL) || (pQualLagData != NULL))
         {
            pQualitativeRawData = (SQP_QualitativeRawData*)malloc(sizeof(SQP_QualitativeRawData));

            pQualitativeRawData->pQualRawData  = pQualData;
            pQualitativeRawData->pQualNames    = NULL;
            pQualitativeRawData->pQualLagData  = pQualLagData;
            pQualitativeRawData->pQualLagNames = NULL;
         }

         /* Make the prediction */
         if (!SQXRunner_DoPredictions(pObj, pOut, pErr, iModelNumber, pObservationRawData, pQualitativeRawData, iCompressed))
         {                        
            szErrString = "DoPredictions failed.";
            goto error_exit;
         }

         /* Clean-up */
         if (pObservationRawData->pRawData)
         {
            SQX_ClearFloatMatrix(pObservationRawData->pRawData);
            free(pRawData);
            pRawData                      = NULL;
            pObservationRawData->pRawData = NULL;
         }
         if (pObservationRawData->pLagData)
         {
            SQX_ClearFloatMatrix(pObservationRawData->pLagData);
            free(pLagData);
            pLagData                      = NULL;
            pObservationRawData->pLagData = NULL;
         }

         if (pObservationRawData)
         {
            free(pObservationRawData);
            pObservationRawData = NULL;
         }

         /* Clean-up the qualitative data */
         if (pQualitativeRawData)
         {
            if (pQualitativeRawData->pQualRawData)
            {
               SQX_ClearStringMatrix(pQualitativeRawData->pQualRawData);
               free(pQualData);
               pQualData                         = NULL;
               pQualitativeRawData->pQualRawData = NULL;
            }
            if (pQualitativeRawData->pQualLagData)
            {
               SQX_ClearStringMatrix(pQualitativeRawData->pQualLagData);
               free(pQualLagData);
               pQualLagData                      = NULL;
               pQualitativeRawData->pQualLagData = NULL;
            }    
            if (pQualitativeRawData)
            {
               free(pQualitativeRawData);
               pQualitativeRawData = NULL;
            }
         }
      }
   }
   SQX_RemoveProject(pObj->mszProjID);
   return;
error_exit:
   if (pRawData)
   {
      SQX_ClearFloatMatrix(pRawData);
      free(pRawData);
      pRawData = NULL;
   }
   if (pLagData)
   {
      SQX_ClearFloatMatrix(pLagData);
      free(pLagData);
      pLagData = NULL;
   }
   if (pQualData)
   {
      SQX_ClearStringMatrix(pQualData);
      free(pQualData);
      pQualData = NULL;
   }
   if (pQualLagData)
   {
      SQX_ClearStringMatrix(pQualLagData);
      free(pQualLagData);
      pQualLagData = NULL;
   }
   fprintf(pErr, "%s\n", szErrString);
   SQX_RemoveProject(pObj->mszProjID);
   return;
}

static int SQXRunner_DoPredictions(SQXRunner* pObj,
                                   FILE* pOut, 
                                   FILE* pErr,
                                   int nModelNumber,
                                   SQP_ObservationRawData *pObservations, 
                                   SQP_QualitativeRawData *pQualitative, 
                                   int iIsCompressed) 
{
   char* szErrString;
   SQX_PredictionHandle    pHandle = NULL;
   /*DllMode eDllMode; */

   /*
    Make the prediction by calling SIMCA-Q with the observation
   */
   if (!SQP_Predict(pObj->mszProjID, nModelNumber, pObservations, pQualitative, iIsCompressed, &pHandle))
   {                        
      szErrString = "Predict failed.";
      goto error_exit;
   }
   fprintf(pOut, "\n");
   /* Get the "static" model data */
   SQXRunner_GetModelParameters(pObj, pOut, pErr, nModelNumber);
   fprintf(pOut, "----------------------------------\n");
   fprintf(pOut, "Predicted Result: \n");

   /* Get the predicted result */
   SQXRunner_GetResults(pObj, pOut, pErr, pHandle, nModelNumber);

   /* Clean up the predictions. */
   SQP_ReleaseHandle(pHandle);

   return 1;
error_exit:
   fprintf(pErr, "%s\n", szErrString);
   return 0;
}

static void SQXRunner_GetResults(SQXRunner* pObj,
                                 FILE* pOut, 
                                 FILE* pErr,
                                 SQX_PredictionHandle pHandle, 
                                 int nModelNumber) 
{
   char*             szErrString;
   int               iNumComp = 0;
   SQX_FloatMatrix   oDModX;
   SQX_FloatMatrix   oT2Range;
   SQX_FloatMatrix   oXVar;


   /*
   Get the number of components for this model
   */
   if (!SQX_GetModelNumberOfComponents(pObj->mszProjID, nModelNumber, &iNumComp))
   {                        
      szErrString = "GetModelNumberOfComponents failed.";
      goto error_exit;
   }

   if (iNumComp > 0)
   {
      SQX_FloatMatrix   oT;
      SQX_FloatMatrix   oContrSSW;

      /*
      Get contribution for the prediction
      */
      if (!SQP_GetPredictedContributionsSSW(pHandle, -1 /* model number, not used */, 0 /*iObs1Ix*/, 1/*iObs2Ix*/, SQX_NoWeight, iNumComp, 1, 0 /*bReconstruct*/, &oContrSSW))
      {
         szErrString = "GetPredictedContributionsSSW failed.";
         goto error_exit;
      }
      else
      {
         fprintf(pOut, "-------------\n");
         fprintf(pOut, "Predicted Contribution SSW:\n");
         SQXRunner_PrintSQX_FloatMatrix(&oContrSSW, pOut, pErr);
         SQX_ClearFloatMatrix(&oContrSSW);
      }

      /*
      Get the predicted T

      GetPredictedT() is not valid for a zero component model. If it's called for with a model
      that has no components it will return 0!
      */
      if (!SQP_GetPredictedT(pHandle, -1 /* model number, not used */, NULL /*pnComponentList*/, &oT))
      {
         szErrString = "GetPredictedT failed.";
         goto error_exit;
      }
      else
      {
         fprintf(pOut, "\nTPS:\n");
         SQXRunner_PrintSQX_FloatMatrix(&oT, pOut, pErr);
         SQX_ClearFloatMatrix(&oT);
      }

   }

   /* Get DModXPS */
   if (!SQP_GetPredictedDModX(pHandle, -1 /* model number, not used */, NULL /*pnComponentList*/, 1 /*bNormalized*/, 0 /*bModelingPowerWeighted*/, &oDModX))
   {
      szErrString = "GetPredictedDModX failed.";
      goto error_exit;
   }
   else
   {
      fprintf(pOut, "DModXPS:\n");
      SQXRunner_PrintSQX_FloatMatrix(&oDModX, pOut, pErr);
      SQX_ClearFloatMatrix(&oDModX);
   }

   /* Get T2RangePS */
   if (!SQP_GetPredictedT2Range(pHandle, -1 /* model number, not used */, 1, iNumComp, &oT2Range))
   {
      szErrString = "GetPredictedT2Range failed.";
      goto error_exit;
   }
   else
   {
      fprintf(pOut, "T2RangePS:\n");
      SQXRunner_PrintSQX_FloatMatrix(&oT2Range, pOut, pErr);
      SQX_ClearFloatMatrix(&oT2Range);
   }

   /* Get XVarPS */
   if (!SQP_GetPredictedXVar(pHandle, -1 /* model number, not used */, 1 /*bUnscaled*/, 1 /*bBackTransformed*/, NULL /*pnColumnXIndices*/, &oXVar))
   {
      szErrString = "GetPredictedXVar failed.";
      goto error_exit;
   }
   else
   {
      fprintf(pOut, "XVarPS:\n");
      SQXRunner_PrintSQX_FloatMatrix(&oXVar, pOut, pErr);
      SQX_ClearFloatMatrix(&oXVar);
   }

   return;
error_exit:
   fprintf(pErr, "%s\n", szErrString);
   return;
}

/* GetModelParameters */
static int SQXRunner_GetModelParameters(SQXRunner* pObj,
                                        FILE* pOut,
                                        FILE* pErr,
                                        int nModelNumber)
{
   const char*       szErrString;
   int               iNumComp = 0;
   int               iNumObs = 0;
   int               iObsIx = 1;
   int               iIsPLSModel = 0;
   float             fPLevel = 0;
   SQX_FloatMatrix   oP;
   SQX_FloatMatrix   oT;
   SQX_FloatMatrix   oT2Range;
   SQX_FloatMatrix   oContrSSW;

   fprintf(pOut, "----------------------------------\n");
   fprintf(pOut, "Model data for model number %d\n", nModelNumber);

   /* Get number of components in the model. */
   if (!SQX_GetModelNumberOfComponents(pObj->mszProjID, nModelNumber, &iNumComp))
   {
      szErrString = "GetModelNumberOfComponents failed.";
      goto error_exit;
   }

   /* Get number of observations in the model. */
   if (!SQX_GetModelNumberOfObservations(pObj->mszProjID, nModelNumber, &iNumObs))
   {
      szErrString = "GetModelNumberOfObservations failed.";
      goto error_exit;
   } 
   fprintf(pOut, "Number of observations is %d.\n", iNumObs);

   fprintf(pOut, "Observation names: ");
   /* Get all observation names. */
   for(iObsIx=1; iObsIx <= iNumObs; ++iObsIx)
   {
      char *szObsName;
      /* Get the observation name. */ 
      if (!SQX_GetModelObservationName(pObj->mszProjID, nModelNumber, iObsIx, 1, &szObsName))
      {
         szErrString = "GetModelObservationName failed.";
         goto error_exit;
      } 
      fprintf(pOut, "%s\t", szObsName);
   }
   fprintf(pOut, "\n");

   if (!SQX_GetModelContributionsSSW(pObj->mszProjID, nModelNumber, 0 /*iObs1Ix*/, 1/*iObs2Ix*/, SQX_NoWeight, iNumComp, 1, 0 /*bReconstruct*/, &oContrSSW))
   {
      szErrString = "GetModelContributionsSSW failed.";
      goto error_exit;
   } 
   else
   {
      fprintf(pOut, "Contribution SSW:\n");
      SQXRunner_PrintSQX_FloatMatrix(&oContrSSW, pOut, pErr);
      SQX_ClearFloatMatrix(&oContrSSW);
   }

   /* Get P for all components in the model. */
   if (!SQX_GetModelP(pObj->mszProjID, nModelNumber, NULL /*pnComponents*/, 0 /*bReconstruct*/, &oP))
   {
      szErrString = "GetModelP failed.";
      goto error_exit;
   }
   else
   {
      fprintf(pOut, "P:\n");
      SQXRunner_PrintSQX_FloatMatrix(&oP, pOut, pErr);
      SQX_ClearFloatMatrix(&oP);
   }

   /* Get T for all components in the model. */
   if (!SQX_GetModelT(pObj->mszProjID, nModelNumber, NULL /*pnComponents*/, &oT))
   {
      szErrString = "GetModelT failed.";
      goto error_exit;
   }
   else
   {
      fprintf(pOut, "T:\n");
      SQXRunner_PrintSQX_FloatMatrix(&oT, pOut, pErr);
      SQX_ClearFloatMatrix(&oT);
   }
   /* Get T2Range */
   if (!SQX_GetModelT2Range(pObj->mszProjID, nModelNumber, 1, iNumComp, &oT2Range))
   {
      szErrString = "GetModelT2Range failed.";
      goto error_exit;
   }
   else
   {
      fprintf(pOut, "T2Range:\n");
      SQXRunner_PrintSQX_FloatMatrix(&oT2Range, pOut, pErr);
      SQX_ClearFloatMatrix(&oT2Range);
   }


   /* Check if the model is a PLS model. */
   if (!SQX_IsModelPLS(pObj->mszProjID, nModelNumber, &iIsPLSModel))
   {
      szErrString = "IsModelPLS failed.";
      goto error_exit;
   }

   if (iIsPLSModel == 1)
   {
      SQX_FloatMatrix   oC;

      /* The following functions are only valid for a PLS model. */

      /* Get C for all components in the model. */
      if (!SQX_GetModelC(pObj->mszProjID, nModelNumber, NULL /*pnComponents*/, &oC))
      {
         szErrString = "GetModelC failed.";
         goto error_exit;
      }
      else
      {
         fprintf(pOut, "C:\n");
         SQXRunner_PrintSQX_FloatMatrix(&oC, pOut, pErr);
         SQX_ClearFloatMatrix(&oC);
      }
   }
   else
   {
      /* The following functions are only valid for a PCA model. */
      SQX_FloatMatrix   oQ2VX;

      /* Get Q2VX */
      if (!SQX_GetModelQ2VX(pObj->mszProjID, nModelNumber, NULL, &oQ2VX))
      {
         szErrString = "GetModelQ2VX failed.";
         goto error_exit;
      }
      else
      {
         fprintf(pOut, "Q2VX:\n");
         SQXRunner_PrintSQX_FloatMatrix(&oQ2VX, pOut, pErr);
         SQX_ClearFloatMatrix(&oQ2VX);
      }
   }
   return 1;
error_exit:
   fprintf(pErr, "%s\n", szErrString);
   return 0;

}

void SQXRunner_PrintSQX_FloatMatrix(SQX_FloatMatrix *pMatrix,
                                         FILE* pOut,
                                         FILE* pErr)
{
   const char* szErrString;
   float fVal;
   int iColIter;
   int iRowIter;

   for (iColIter = 0; iColIter < SQX_GetNumColumnsInFloatMatrix(pMatrix); ++iColIter)
   {
      for(iRowIter = 0; iRowIter < SQX_GetNumRowsInFloatMatrix(pMatrix); ++iRowIter)
      {
         /* Print the results tab separated to the file */
         if (!SQX_GetDataFromFloatMatrix(pMatrix, iRowIter + 1, iColIter + 1, &fVal))
         {
            szErrString = "SQX_GetDataFromFloatMatrix failed.";
            goto error_exit;
         }

         fprintf(pOut, "%f\t",fVal);
      }
      fprintf(pOut, "\n");
   }
   fprintf(pOut, "\n");

   return;
error_exit:
   fprintf(pErr, "%s\n", szErrString);
   return;
}
int UmFileReader(SQXRunner* pObj, FILE* pErr)
{
   unsigned  iPos, iCol=1;
   size_t  count;
   int iMaxCount;
   char *tmpBuffer;
   unsigned iNumCurCols=0;

   pObj->iFileNumRows = 0;
   pObj->iFileNumCols = 0;

   /*Read the input file. Decide the file size*/
   iPos = fseek(pObj->mInputFile, 0, SEEK_END);
   if (iPos)
      goto error_file_exit;
   iMaxCount = ftell(pObj->mInputFile);
   iPos = fseek(pObj->mInputFile, 0, SEEK_SET);
   if (iPos)
      goto error_file_exit;
   pObj->szFileBuffer = (char*)malloc(iMaxCount+1);
   if (pObj->szFileBuffer==NULL)
      goto error_file_exit;

   count = fread(pObj->szFileBuffer, sizeof( char ), iMaxCount, pObj->mInputFile );
   if (ferror( pObj->mInputFile ))
   {
      free(pObj->szFileBuffer);
      pObj->szFileBuffer = NULL;
      goto error_file_exit;
   }
   pObj->szFileBuffer[count]='\0';
   /*Decide the size of the data.*/
   iPos=0;

   while (iPos < count)
   {
      const unsigned iNext = iPos+1;
      /* ignore empty rows */
      if (pObj->szFileBuffer[iPos]=='\n' && (pObj->szFileBuffer[iNext]=='\n' || pObj->szFileBuffer[iNext]=='\0'))
      {
         iPos++;
         continue;
      }
      if (pObj->szFileBuffer[iPos]=='\t')
         iNumCurCols++;
      if (pObj->szFileBuffer[iPos]=='\n')
      {
         iNumCurCols++;
         if (pObj->iFileNumCols>0 && iNumCurCols!=pObj->iFileNumCols)
         {
            free(pObj->szFileBuffer);
            pObj->szFileBuffer = NULL;
            goto error_file_exit;
         }
         pObj->iFileNumCols=iNumCurCols;
         pObj->iFileNumRows++;
         iNumCurCols=0;
      }
      iPos++;
   }
   /* 
   Read the first line of the file, it should be the variable names.
   The file should be tab separated, search for the first tab.
   Separate all variable names from the tabs and store them in a vector. 
   */
   if (!SQX_InitStringVector(&pObj->mstrVariableNames, pObj->iFileNumCols))
   {
      fprintf(pErr, "Unable to init vector for the variablenames.\n");
      return 0;
   }
   tmpBuffer=&pObj->szFileBuffer[0];
   for (iCol = 1; iCol <= pObj->iFileNumCols; iCol++)
   {
      iPos=0;
      while (tmpBuffer[iPos]!='\t' && tmpBuffer[iPos]!='\n' && tmpBuffer[iPos]!='\0')
         iPos++;

      tmpBuffer[iPos]='\0';

     SQX_SetStringInVector(&pObj->mstrVariableNames, iCol, tmpBuffer);

     tmpBuffer=&tmpBuffer[iPos+1];
   }
   return 1;
error_file_exit:
   fprintf(pErr, "File Reading error.\n");
   return 0;
}

int GetQuantitativeData(SQXRunner* pObj, SQX_StringVector *pstrVariableNames, FILE* pErr, SQX_FloatMatrix *pVariableMat)
{
   SQX_IntVector oIndexVector;
   int iIndexVectorSize;
   char *tmpBuffer;
   unsigned  iPos, iCol=1, iRow=1;

   if (!GetIndexVectorFromStrings(pObj, pstrVariableNames, pErr, &oIndexVector))
   {
      return 0;
   }
   iIndexVectorSize = SQX_GetIntVectorSize(&oIndexVector);
   if (SQX_GetNumStringsInVector(pstrVariableNames) < 1) 
   {
      return 0;
   }
   if (iIndexVectorSize < 1)
   {
      fprintf(pErr, "Can not find the quantitative variable names in the file.\n");
      return 0;
   }
   if (!SQX_InitFloatMatrix(pVariableMat, pObj->iFileNumRows, iIndexVectorSize))
   {
      fprintf(pErr, "SQX_InitFloatMatrix failed.\n");
      return 0;
   }
   /* Move the file pointer to the beginning of the file. */
   iPos = fseek( pObj->mInputFile, 0, SEEK_SET );
   if (iPos)
      goto error_file_exit;

   /* Read the first line, that is the variable names, we are not interested in those now. */
   tmpBuffer=&pObj->szFileBuffer[0];
   iPos=0;
   for (iCol = 1; iCol <= pObj->iFileNumCols; iCol++)
   {
      iPos = 0;
      while (tmpBuffer[iPos]!='\t' && tmpBuffer[iPos]!='\n' && tmpBuffer[iPos]!='\0')
         iPos++;

      tmpBuffer=&tmpBuffer[iPos+1];
   }

   /* Read one line at a time. */
   for (iRow = 1; iRow <= pObj->iFileNumRows; iRow++)
   {
      int      iIndex = 1;
      int      iVectorIndex = 0;
      int      iDataFromVector = 0;
      for (iCol = 1; iCol <= pObj->iFileNumCols; iCol++)
      {
         iPos=0;
         while (tmpBuffer[iPos]!='\t' && tmpBuffer[iPos]!='\n' && tmpBuffer[iPos]!='\0')
            iPos++;

         tmpBuffer[iPos]='\0';

         for (iVectorIndex = 1; iVectorIndex <= iIndexVectorSize; ++iVectorIndex)
         {
            SQX_GetDataFromIntVector(&oIndexVector, iVectorIndex, &iDataFromVector);
            if (iIndex == iDataFromVector)
            {
               /* This is the value we want. Convert it*/
               float fVal = (float)atof(tmpBuffer);
               if (!SQX_SetDataInFloatMatrix(pVariableMat, iRow, iVectorIndex, fVal))
               {
                  SQX_ClearIntVector(&oIndexVector);
                  fprintf(pErr, "SQX_SetDataInFloatMatrix failed.\n");
                  return 0;
               }
               break;
            }
         }
         ++iIndex;
         tmpBuffer=&tmpBuffer[iPos+1];
      }
   } 
   SQX_ClearIntVector(&oIndexVector);
   return 1;

error_file_exit:
   fprintf(pErr, "File Reading error");
   return 0;
}

int GetQualitativeData(SQXRunner* pObj, SQX_StringVector *pstrVariableNames, FILE* pErr, SQX_StringMatrix *pQualVarMat)
{
   SQX_IntVector oIndexVector;
   int iIndexVectorSize;
   char *tmpBuffer;
   unsigned  iPos, iCol=1, iRow=1;
   if (!GetIndexVectorFromStrings(pObj, pstrVariableNames, pErr, &oIndexVector))
   {
      return 0;
   }
   iIndexVectorSize = SQX_GetIntVectorSize(&oIndexVector);
   if (SQX_GetNumStringsInVector(pstrVariableNames) < 1) 
   {
      return 0;
   }
   if (iIndexVectorSize < 1)
   {
      fprintf(pErr, "Can not find the qualitative variable names in the file.\n");
      return 0;
   }
   if (!SQX_InitStringMatrix(pQualVarMat, pObj->iFileNumRows, iIndexVectorSize))
   {
      fprintf(pErr, "SQX_InitStringMatrix failed.\n");
      return 0;
   }
   /* Move the file pointer to the beginning of the file. */
   iPos = fseek( pObj->mInputFile, 0, SEEK_SET );
   if (iPos)
      goto error_file_exit;

   /* Read the first line, that is the variable names, we are not interested in those now. */
   tmpBuffer=&pObj->szFileBuffer[0];
   iPos=0;
   for (iCol = 1; iCol <= pObj->iFileNumCols; iCol++)
   {
      iPos = 0;
      while (tmpBuffer[iPos]!='\t' && tmpBuffer[iPos]!='\n' && tmpBuffer[iPos]!='\0')
         iPos++;
      tmpBuffer=&tmpBuffer[iPos+1];
   }

   /* Read one line at a time. */
   for (iRow = 1; iRow <= pObj->iFileNumRows; iRow++)
   {        
      int      iIndex = 1;
      int      iVectorIndex = 0;
      int      iDataFromVector = 0;
      for (iCol = 1; iCol <= pObj->iFileNumCols; iCol++)
      {
         iPos=0;
         while (tmpBuffer[iPos]!='\t' && tmpBuffer[iPos]!='\n' && tmpBuffer[iPos]!='\0')
            iPos++;

         tmpBuffer[iPos]='\0';

         for (iVectorIndex = 1; iVectorIndex <= iIndexVectorSize; ++iVectorIndex)
         {
            SQX_GetDataFromIntVector(&oIndexVector, iVectorIndex, &iDataFromVector);
            if (iIndex == iDataFromVector)
            {
               /* This is the value we want. */
               if (!SQX_SetStringInMatrix(pQualVarMat, iRow, iVectorIndex, tmpBuffer))
               {
                  fprintf(pErr, "SQX_SetStringInMatrix failed.\n");
                  return 0;
               }
               break;
            }
         }
         ++iIndex;
         tmpBuffer=&tmpBuffer[iPos+1];
      }
   } 
   SQX_ClearIntVector(&oIndexVector);
   return 1;
error_file_exit:
   fprintf(pErr, "File Reading error");
   return 0;
}

int GetIndexVectorFromStrings(SQXRunner* pObj, SQX_StringVector *pNames, FILE* pErr, SQX_IntVector *pIndexVector)
{
   const char *szVarName;
   const char *szStringIt;
   int iIndex;
   int iVarName=1;
   int iCheck = 0;
   int iRowCount=1;
   int init=0;

   /* Go through all variable names. Count how many there are matching in file. */
   for (iVarName = 1; iVarName <= SQX_GetNumStringsInVector(pNames); ++iVarName)
   {
      SQX_GetStringFromVector(pNames,iVarName, &szVarName);

      /* Find this name in the vector with variable names in the file. */
      for (iIndex=1; iIndex <= SQX_GetNumStringsInVector(&pObj->mstrVariableNames); iIndex++)
      {
         SQX_GetStringFromVector(&pObj->mstrVariableNames, iIndex, &szStringIt);
         if (strcmp(szStringIt, szVarName) == 0)
         {
            init++;
            break;
         }
      }
   }
   if (init == 0)
   {
      fprintf(pErr, "Could not find any matching variables in file.\n");
      return 0;
   }
   /* Init the return vector */
   if (!SQX_InitIntVector(pIndexVector, init))
   {
      fprintf(pErr, "SQX_InitIntVector failed.\n");
      return 0;
   }
   /* Go through all variable names. */
   for (iVarName = 1; iVarName <= SQX_GetNumStringsInVector(pNames); ++iVarName)
   {
      iCheck = 0;
      SQX_GetStringFromVector(pNames,iVarName, &szVarName);

      /* Find this name in the vector with variable names in the file. */
      for (iIndex=1; iIndex <= SQX_GetNumStringsInVector(&pObj->mstrVariableNames); iIndex++)
      {
         SQX_GetStringFromVector(&pObj->mstrVariableNames, iIndex, &szStringIt);
         if (strcmp(szStringIt, szVarName) == 0)
         {
            /* This is the name, store the index of the variable in the vector. */
            if (!SQX_SetDataInIntVector(pIndexVector, iRowCount, iIndex))
            {
               SQX_ClearIntVector(pIndexVector);
               fprintf(pErr, "SQX_SetDataInIntVector failed.\n");
               return 0;
            }
            iRowCount++;
            iCheck = 1;
            break;
         }
      }
      if (iCheck == 0)
      {
         /* The string was not found. */
         char strError[100];
         strcpy(strError, "Variable ");
         strcat(strError, szVarName);
         strcat(strError, " could not be found.");
         fprintf(pErr, "%s\n", strError);
         break;
      }
   }
   if (iCheck == 0)
   {
      SQX_ClearIntVector(pIndexVector);
      return 0;
   }
   return 1;
}
